(define (new-constraint-network)
  (display "\nCreating a new constraint network.\n")

  (list 'constraint-network
	(list 'variables)
	(list 'constraint-instances)
	)
  )


(define (constraint-network::variables constraint-network) (second constraint-network))
(define (constraint-network::constraint-instances constraint-network) (third constraint-network))

;;;;;;;;;;;;;;
;; adds a new variable to the network
;;
;; @param constraint-network = a constraint network made with (new-constraint-network)
;; @param variable-name = a symbol or list, representing the name of a variable in the constriant network
;; @param domain = a list, each element of which is a value to be placed in the domain of variable 
;; @returns Undefined
;; @sideeffect modifies the constraint-network to add a new variable.
(define (constraint-network::insert-variable! constraint-network variable-name variable-domain)
  ;; first we get the variables list (note that it starts off with an
  ;; list labeling symbol: 'variables, which we'll want to ignore
  (let ((variables (constraint-network::variables constraint-network)))
    ;; next we'll see if the network already has a variable with the specified domain
    (let ((varpair (assoc variable-name (skip-list-label variables))))
      (if varpair        
        (set-cdr! varpair variable-domain) ;; the variable did exist
        (set-cdr! variables (cons (cons variable-name variable-domain) (skip-list-label variables))) ;; the variable is new 
      )
    )  
  )
)

(define constraint-network::set-variable-domain! constraint-network::insert-variable!) 

;;;;;;;;;;;;;;
;; looks up a variable's domain
;;
;; @param constraint-network = a constraint network made with (new-constraint-network)
;; @param variable-name = a symbol or list, representing the name of a variable in the constriant network
;; @returns a mutable scheme list containing every value in the domain of variable
(define (constraint-network::get-variable-domain constraint-network variable-name)
  (let ((variables (constraint-network::variables constraint-network)))    
    (cdr (assoc variable-name (skip-list-label variables)))
  )
)

;;;;;;;;;;;;;;
;; lists network variables
;;
;; @param constraint-network = a constraint network made with (new-constraint-network)
;; @returns a list containing the name of every variable in the network
(define (constraint-network::variable-names constraint-network)
  (let ((variables (constraint-network::variables constraint-network)))
    (map first (skip-list-label variables))
  )
)

;;;;;;;;;;;;;;
;; adds a new constraint instance to the network
;;
;; @param constraint-network = a constraint network made with (new-constraint-network)
;; @param head = a variable name.
;; @param tail = a variable name.
;; @param description = a description of this constraint (for debugging)
;; @param predicate = a function of the form (lambda head-value tail-value parameters).  When this constraint is
;;                    checked, the predicate will be evaluated with a value from head's domain as the
;;                    head-value argument, and a value from the tail's domain as the tail-value argument.
;;                    The predicate should return #t if the head-value and tail-value satisfy the constraint,
;;                    otherwise it should return #f.
;; @param paremeters = an assoc list; that is, a list of pairs, where the car of each pair is the name (a scheme symbol) of a parameter,
;;                    and the cdr of the pair is the value of the parameter.  The purpose of the parameters list is
;;                    to allow a predicate to be reused in many constraint-instances, but to allow some level of
;;                    customization for each instance.
;; @returns Undefined
;; @sideeffect modifies the constraint-network to add a new constraint
;; @example (let ((myPredicate (lambda head-value tail-value parameters) 
;;                               (> (+ head-value tail-value) 
;;                                  (lookup-parameter 'threshold parameters))))
;;            (begin
;;              (constraint-network::insert-constraint-instance! myNetwork myVarA myVarB "Make sure A & B sum to greater than 5" myPred '((threshold . 5)))
;;              (constraint-network::insert-constraint-instance! myNetwork myVarB myVarC "Make sure B & C sum to greater than 9" myPred '((threshold . 9)))
;;             )
;;           )
(define (constraint-network::insert-constraint-instance! constraint-network head tail description predicate parameters)
  (let ( (constraint-instance (list 'constraint-instance head tail description predicate parameters))
         (constraint-instances (constraint-network::constraint-instances constraint-network)))
      (set-cdr! constraint-instances (cons constraint-instance (skip-list-label constraint-instances)))         
  )
)

;;;;;;;;;;;;;
;; lookup a parameter in a constraint-instance's parameter list
;;
;; @param parameter-name = a parameter name
;; @param parameters = an assoc list; that is, a list of pairs, where the car of each pair is the name (a scheme symbol) of a parameter,
;;                     and the cdr of the pair is the value of the parameter.
;; @returns the value associated with the specified parameter-name in the parameters list
(define (lookup-parameter parameter-name parameters)
  (cdr (assoc parameter-name parameters))
)

(define (constraint-instance::head constraint-instance) 
(second constraint-instance))
(define (constraint-instance::tail constraint-instance) 
(third constraint-instance))
(define (constraint-instance::description constraint-instance) 
(fourth constraint-instance))
(define (constraint-instance::predicate constraint-instance) 
(fifth constraint-instance))
(define (constraint-instance::parameters constraint-instance) 
(sixth constraint-instance))
  
  

