;=============================================================================
;===============             VARIABLE ASSIGNMENTS           ==================
;=============================================================================

;;;;;;;;;;;;;;;
;;  Create a new variable assignment
;;
;; @param assignment-variable = a variable name
;; @param assignment-value = the value assigned to that variable
;; @return a record of the variable assignment.  Lists of these records are
;;     returned by the constraint search algorithms when they are successful
(define (new-assignment assignment-variable assignment-value)
  (cons assignment-variable assignment-value)
)




;=============================================================================
;===============        CONSTRAINT SEARCH STRATEGIES        ==================
;=============================================================================

;;;;;;;;;;;;;;;;
;;  Create a new constraint-search strategy
;;
;; @param pick-variable = a function of the form (lambda constraint-network variables-to-choose-from)
;;                        where constraint-network is a constraint network data structure and 
;;                        variables-to-choose-from is a list of variable names.  The function
;;                        should return a variable name from variables-to-choose-from, indicating
;;                        which variable should have a value assigned to it next.
;;                        See: constraint-search-strategy::pick-variable::pick-first
;; @param propagation-strategy = a propagation stragegy, created with new-propagation-strategy.  This
;;                        strategy is used for all calls to propagate-constraints! that are performed
;;                        in the course of the search
;;                        See: *propagate-constraints-strategy::full*
;;                          and *propagate-constraints-strategy::domains-of-one*
;;                          and *propagate-constraints-strategy::never*
;; @param have-enough-successes? = a function of the form (lambda successes-so-far), where 
;;                        successes-so-far is a list, whose elements are lists of variable assignments
;;                        representing a compatible assignment of all variables in the network.
;;                        This function should return #t if enough successful assignments have been
;;                        found, or #f if the search algorithm should keep searching for more. 
;;                        See: constraint-search-strategy::have-enough-successes?::one
;;                          and constraint-search-strategy::have-enough-successes?::all
(define (new-constraint-search-strategy pick-variable propagation-strategy have-enough-successes?)
  (list 'constraint-search-strategy
    pick-variable
    propagation-strategy
    have-enough-successes?
  )
)

(define (constraint-search-strategy::have-enough-successes?::all successes-so-far)
  #f
)

(define (constraint-search-strategy::pick-variable::pick-first constraint-net vars-to-choose-from)
  (first vars-to-choose-from)
  )

(define *constraint-search-strategy::full*
  (new-constraint-search-strategy
   constraint-search-strategy::pick-variable::pick-first
   *propagate-constraints-strategy::full*
   constraint-search-strategy::have-enough-successes?::one
   )
  )

(define *constraint-search-strategy::only-check-assignments*
  (new-constraint-search-strategy
    constraint-search-strategy::pick-variable::pick-first
    *propagate-constraints-strategy::never*
    constraint-search-strategy::have-enough-successes?::one
  )
)
    
(define *constraint-search-strategy::domains-of-one*
  (new-constraint-search-strategy
    constraint-search-strategy::pick-variable::pick-first
    *propagate-constraints-strategy::domains-of-one*
    constraint-search-strategy::have-enough-successes?::one
  )
)

(define *constraint-search-strategy::only-check-assignments/all*
  (new-constraint-search-strategy
    constraint-search-strategy::pick-variable::pick-first
    *propagate-constraints-strategy::never*
    constraint-search-strategy::have-enough-successes?::all
  )
)
    
(define *constraint-search-strategy::domains-of-one/all*
  (new-constraint-search-strategy
    constraint-search-strategy::pick-variable::pick-first
    *propagate-constraints-strategy::domains-of-one*
    constraint-search-strategy::have-enough-successes?::all
  )
)

(define *constraint-search-strategy::full/all*
  (new-constraint-search-strategy
    constraint-search-strategy::pick-variable::pick-first
    *propagate-constraints-strategy::full*
    constraint-search-strategy::have-enough-successes?::all
  )
)
    
(define (constraint-search-strategy::pick-variable search-strategy)
  (second search-strategy)
)

(define (constraint-search-strategy::propagation-strategy search-strategy)
  (third search-strategy)
)

(define (constraint-search-strategy::have-enough-successes? search-strategy)
  (fourth search-strategy)
)

;=============================================================================
;===============        CONSTRAINT SEARCH ALGORITHM         ==================
;=============================================================================


;; @returns consistent list of assignments, or #f if none exists
(define (constraint-search search-strategy constraint-network)
  (constraint-search-next-variable
    search-strategy 
    constraint-network 
    (constraint-network::variable-names constraint-network) 
    '() ;; assignments-so-far (haven't made any yet)
    '(successes) ;; successes-so-far (haven't found any yet)
  )
)

(define (constraint-search-next-variable search-strategy constraint-network variables-to-be-assigned assignments-so-far successes-so-far)
  (if (null? variables-to-be-assigned)
    ;; assignments-so-far contains an assigment to every variable
    (cons 'successes (cons assignments-so-far (skip-list-label successes-so-far)))

    ;; pick an unassigned variable and try to assign it a value 
    (let* ((assignment-variable ((constraint-search-strategy::pick-variable search-strategy) constraint-network variables-to-be-assigned))           
           (assignment-domain (constraint-network::get-variable-domain constraint-network assignment-variable))
           (remaining-variables (delete assignment-variable variables-to-be-assigned)))
      (constraint-search-next-assignment 
        search-strategy
        constraint-network
        remaining-variables
        assignments-so-far
        successes-so-far
        assignment-variable
        assignment-domain
      )
    )
  )
)

(define (constraint-search-next-assignment search-strategy constraint-network 
          variables-to-be-assigned assignments-so-far successes-so-far 
          assignment-variable assignment-values-to-try)
  (if (null? assignment-values-to-try)
    ;; we've exhausted all the values in this variable's domain.   We have to backtrack
    successes-so-far
    
    ;; pick a value and try assigning it in the network.
    ;; we'll work with a copy of the constraint-network, so that we can backtrack
    ;; without having to worry about having clobbered the network state
    (let* ((working-copy-of-network (clone constraint-network))
        (assignment-value (first assignment-values-to-try))
        (remaining-values-to-try (rest assignment-values-to-try))
        (constraint-propagation-todo-list '(todo)))
      (begin
        ;; make the assignment in the network
        (constraint-network::set-variable-domain! working-copy-of-network assignment-variable (list assignment-value))
        ;; find out which constraints we need to check to validate that assignment
        (propagate-constraints::cascade! working-copy-of-network constraint-propagation-todo-list assignment-variable)
        ;; check the constraints!
        (let* ((propagation-successful (propagate-constraints::do-todo-list!
                                        (constraint-search-strategy::propagation-strategy search-strategy)
                                        working-copy-of-network
                                        constraint-propagation-todo-list))
               ;; if we successfully assigned, then we can recursively try more assignments.  Hopefully 
               ;; recursing will discover some new compatible sets of variable assignments (aka, "successes"!) 
               (updated-successes-so-far
                  (if propagation-successful
                  ;; no conflicts were found during propagation!  we can try doing more assignments 
                    (constraint-search-next-variable
                      search-strategy
                      working-copy-of-network 
                      variables-to-be-assigned
                      (cons (new-assignment assignment-variable assignment-value) assignments-so-far)
                      successes-so-far
                    )
                    
                    ;; there was a conflict, so we'll have to just move on to other values
                    successes-so-far
                  )
                ))
          
          (if ((constraint-search-strategy::have-enough-successes? search-strategy) updated-successes-so-far)
            ;; we've found enough successes, no need to try any more.
            updated-successes-so-far
            
            ;; we still need more successes
            (constraint-search-next-assignment
              search-strategy
              constraint-network
              variables-to-be-assigned
              assignments-so-far
              updated-successes-so-far
              assignment-variable
              remaining-values-to-try
            )
          )
        )
      )
    )
  )
)

