(load "useful.scm")
(load "constraint-network.scm")
(load "propagate-constraints.scm")

;;Basic Constraint Package Example
;;See constraints-notes.txt for an overview of the constraint package,
;;and pointers to its guts (which have more precise comments).
;;
;;This example constructs a net with three variables, each with simple
;;finite numerical domains, and runs one full propagation step on them
;;(that attempts to rule out all values from each variables' domain
;;that are clearly inconsistent with the constraints on that
;;variable). It then returns the state of the constraint network after
;;propagation, showing which domain possibilities have been removed.
;;
;;In this case, for example, if vara is 4, the constraint a + b >  11
;;cannot be satisfied. The full propagator discovers this during its
;;first run. The full propagator does *not* identify a single solution
;;to the search problem; it simply eliminates things that are clearly
;;inconsistent. For example, if c is 3, b could be 6, but if c is 4,
;;b could be 5. Full propagation cannot figure out which single values
;;to pick.

;;When evaluated on a network, this procedure adds a variable named
;;vara with a domain of 23 and 4, varb with the domain of 3, 5 and 6,
;;and so-on.

(define (create-variables-simple-net net)
  (constraint-network::insert-variable! net 'vara '(23 4))
  (constraint-network::insert-variable! net 'varb '(3 5 6))
  (constraint-network::insert-variable! net 'varc '(1 2 3 4))
)

;;This procedure tests if the values at the head and the tail of the
;;constraint satisfy the relation (head-value + tail-value) > threshold
;;where threshold is specified as an association list. It is used
;;to implement the sum>threshold constraints.

(define (sum>threshold head-value tail-value parameters)
  (> (+ head-value tail-value) (lookup-parameter 'threshold parameters))
)

;;This procedure adds sum>threshold arcs between variables a and b (with
;;threshold 11) and variables b and c (with threshold 8) to the constraint
;;network passed in.

(define (create-arcs-simple-net net)
  (constraint-network::insert-constraint-instance! net 'vara 'varb
						   "a+b > 11"
						   sum>threshold
						   '((threshold . 11))
						   )
  (constraint-network::insert-constraint-instance! net 'varb 'varc
						   "b+c > 8"
						   sum>threshold
						   '((threshold . 8))
						   )
  )

;;This procedure creates a new empty network, adds the variables a,b,c to it
;;with the domains specified above, adds the constraint arcs specified above,
;;runs full propagation on the net once, then returns the state of the
;;variables in the net.

(define (go-simple-net)
  (let ((simple-net (new-constraint-network))
	)
    (create-variables-simple-net simple-net)
    (create-arcs-simple-net simple-net)
    (propagate-constraints! *propagate-constraints-strategy::full* simple-net)
    (constraint-network::variables simple-net)
    )
  )

;;This permits easy viewing of the results.

(pp (go-simple-net))
