;=============================================================================
;===============      PROPAGATE-CONSTRAINTS-STRATEGY    ======================
;=============================================================================

;;;;;;;;;;;;;;;;
;; creates a propagate-constraints-strategy.  Use the strategy by passing it to propagate-constraints!
;;
;; @param domain-change-cascade = a predicate of the form (lambda checked-variable old-domain new-domain), which
;;                                should return #t whenever changing the domain of checked-variable from 
;;                                old-domain to new-domain should cause constraints connected to checked-variable
;;                                to be rechecked.
;; @returns a propagate-constraints-strategy suitable for passing to propagate-constraints!
(define (propagate-constraints::make-strategy domain-change-cascade?)
  (list 'propagate-constraints-strategy domain-change-cascade?)
)

;;;;;;;;;;;;;;;;
;; @accessor
;; @returns a predicate of the form (lambda checked-variable old-domain new-domain), which
;;              should return #t whenever changing the domain of checked-variable from 
;;              old-domain to new-domain should cause constraints connected to checked-variable
;;              to be rechecked.
(define (propagate-constraints-strategy::domain-change-cascade? strategy) (second strategy))

;;;;;;;;;;;;;;;;
;; this strategy causes constraints to be rechecked whenever one of their variables' domains
;; decrease
;;
;; this is the most "complete" pure constraint-propagation strategy, though search is still 
;; required in some cases to find consistent sets of variable assignments.  
(define *propagate-constraints-strategy::full* 
 (propagate-constraints::make-strategy
   (lambda (checked-variable old-domain new-domain) (< (length new-domain) (length old-domain)))
 )
)

;;;;;;;;;;;;;;;;;
;; this strategy causes constraints to be rechecked only when one of their variables' domains
;; decrease to a size of 1.
;;
;; Warning: this strategy is not "complete", in that it may not eliminate all values that
;; *full-constraint-propagation-strategy* would.
(define *propagate-constraints-strategy::domains-of-one*
  (propagate-constraints::make-strategy
    (lambda (checked-variable old-domain new-domain) 
      (and (< (length new-domain) (length old-domain))
        (equal? (length new-domain) 1)))
  )
)

;;;;;;;;;;;;;;;;;
;; this strategy causes constraints to never be rechecked.
;;
;; Warning: this strategy is not "complete", in that it may not eliminate all values that
;; *full-constraint-propagation-strategy* would.
(define *propagate-constraints-strategy::never*
  (propagate-constraints::make-strategy
    (lambda (checked-variable old-domain new-domain) #f)
  )
)

;============================================================================
;===============      PROPAGATE-CONSTRAINTS-TODO-ITEM    ====================
;============================================================================

;;;;;;;;;;;;;;;;;;
;; @param constraint-instance = a constraint instance
;; @param direction = either 'forward or 'backward.  If 'forward, then values 
;;                    in the head domain are in danger or removal.  If 
;;                    backward, then values in the tail domain are in danger
;;                    of removal.
;; @returns a todo-item for the propagates constrainst todo-list
(define (propagate-constraints::new-todo-item constraint-instance direction)
  (list constraint-instance direction)
)

;;;;;;;;;;;;
;; @accessor
;; @returns a constraint instance
(define (propagate-constraints-todo-item::constraint-instance todo-item) (first todo-item))

;;;;;;;;;;;;
;; @accessor
;; @returns either 'forward or 'backward.  If 'forward, then values 
;;                    in the head domain are in danger or removal.  If 
;;                    backward, then values in the tail domain are in danger
;;                    of removal.
(define (propagate-constraints-todo-item::direction todo-item) (second todo-item))  




;=============================================================================
;================            PROPAGATE-CONSTRAINTS        ==================== 
;=============================================================================

;;;;;;;;;;;
;; this procedure performs the PURE CONSTRAINT PROPAGATION algorithm on
;; constraint-network.  The algorithm's execution is tailored by the strategy.
;;
;; @param strategy = defines the strategy to be used when propagating constraints, made with (propagate-constraints::make-strategy)
;; @param constraint-network = a constraint network made with (new-constraint-network)
;; @returns #f if a variable developed an empty domain, #t otherwise
;; @sideeffect runs constraint propagation on the constraint-network (does NOT do variable assignment searching)
;; @sideeffect modifies constraint-network (reduces variable domains)
(define (propagate-constraints! strategy constraint-network)
  (let ((todo-list (propagate-constraints::generate-todo-list constraint-network)))
    (propagate-constraints::do-todo-list! strategy constraint-network todo-list)
  )
)

;;;;;;;;;;;;
;; Computes a list of all the work that must be done to complete propagate-constaints! (that is,
;; every constriant-instance needs to be checked in both the forward and backward direction).
;;
;; @param constraint-network = a constraint network made with (new-constraint-network)
;; @returns a list of todo-items, one for checking both directions of each constraint instance in the network
(define (propagate-constraints::generate-todo-list constraint-network)
  (let ((constraint-instances (constraint-network::constraint-instances constraint-network)))
    (cons 'todo
      (append 
        (map 
          (lambda (constraint-instance) (propagate-constraints::new-todo-item constraint-instance 'forward))
          (skip-list-label constraint-instances))
        (map
          (lambda (constraint-instance) (propagate-constraints::new-todo-item constraint-instance 'backward))
          (skip-list-label constraint-instances))
      )
    )
  )        
)

;;;;;;;;;;;;
;; this routine takes each piece of work off the todo list and delegates it to check-constraint-instance!
;;
;; @param strategy = defines the strategy to be used when propagating constraints
;; @param constraint-network = a constraint network made with (new-constraint-network)
;; @param todo-list = a list of todo-items, describing the work that still needs to be done.
;; @returns #f if a variable developed an empty domain, #t otherwise
;; @sideeffect modifies constraint-network (reduces variable domains)
;; @sideeffect modifies todo-list (adds and removes work)
(define (propagate-constraints::do-todo-list! strategy constraint-network todo-list)
   (if (null? (skip-list-label todo-list))
    #t
    (let ((current-todo-item (second todo-list)))
      (if 
       (propagate-constraints::check-constraint-instance! 
	strategy
	constraint-network
	todo-list
	(propagate-constraints-todo-item::constraint-instance current-todo-item)
	(propagate-constraints-todo-item::direction current-todo-item)
	)
       
        (begin          
          (delete! current-todo-item todo-list)
          (propagate-constraints::do-todo-list! strategy constraint-network todo-list)
        )
        #f
      )
    )
  )
)

;;;;;;;;;;;;
;; checks a constaint-instance to see if any values need to be eliminated from a variable's domain
;;
;; @param strategy = defines the strategy to be used when propagating constraints
;; @param constraint-network = a constraint network made with (new-constraint-network)
;; @param todo-list = a list of todo-items, describing the work that still needs to be done.
;; @param constraint-instance = the constraint-instance to check
;; @param direction = either 'forward or 'backward.  If 'forward, then values 
;;                    in the head domain are in danger or removal.  If 
;;                    backward, then values in the tail domain are in danger
;;                    of removal.
;; @returns #f if checking this constraint-instance caused a variable to develop an empty domain, #t otherwise
;; @sideeffect modifies constraint-network (reduces variable domains)
;; @sideeffect modifies todo-list (adds new work)
(define (propagate-constraints::check-constraint-instance! strategy constraint-network todo-list constraint-instance direction)
  
  (let* ((head-variable (constraint-instance::head constraint-instance))
        (tail-variable (constraint-instance::tail constraint-instance)))
    (let*(
	  (head-domain (constraint-network::get-variable-domain constraint-network head-variable))
	  (tail-domain (constraint-network::get-variable-domain constraint-network tail-variable))
	  (predicate (constraint-instance::predicate constraint-instance))
	  (parameters (constraint-instance::parameters constraint-instance))
	  )
    
    (if (eq? direction 'forward)
      ;; forward dirction: filter the head-domain
      (let ((new-head-domain
	     (keep-matching-items 
	      head-domain 
                (lambda (head-value)
                  (any? 
                    tail-domain
                    (lambda (tail-value) (predicate head-value tail-value parameters))
                  )
                )
              )
            ))
        (propagate-constraints::check-for-domain-change! strategy constraint-network todo-list head-variable head-domain new-head-domain)
      )
    
      ;; backward direction: filter the tail-domain
      (let ((new-tail-domain
              (keep-matching-items 
                tail-domain 
                (lambda (tail-value)
                  (any? 
                    head-domain
                    (lambda (head-value) (predicate head-value tail-value parameters))
                  )
                )
              )
            ))
        (propagate-constraints::check-for-domain-change! strategy constraint-network todo-list tail-variable tail-domain new-tail-domain)
      )
    ))
  )
)


;;;;;;;;;;;;;;;
;; this routine is called to enact (potential) changes to a variable's domains, as a side effect of checking a constraint instance
;;
;; @param strategy = defines the strategy to be used when propagating constraints
;; @param constraint-network = a constraint network made with (new-constraint-network)
;; @param todo-list = a list of todo-items, describing the work that still needs to be done.
;; @param checked-variable = the variable whose domain is under consideration
;; @param old-domain = the current domain for this variable
;; @param new-domain = the proposed new domain
;; @assumption this procedure assumes that checking for changes in the domain length is equivalent
;;             to checking for changes in the domain (this assumption is fine as long as noone is
;;             sneakily inserting values into domains)
;; @returns #f if the new-domain is empty, #t otherwise
;; @sideeffect modifies constraint-network (reduces variable domains)
;; @sideeffect modifies todo-list (adds new work)
(define (propagate-constraints::check-for-domain-change! strategy constraint-network todo-list checked-variable old-domain new-domain)
  (if (equal? old-domain new-domain)
    (not (null? new-domain))
    (begin
      (constraint-network::set-variable-domain! constraint-network checked-variable new-domain)
      (if ((propagate-constraints-strategy::domain-change-cascade? strategy) checked-variable old-domain new-domain)
        (propagate-constraints::cascade! constraint-network todo-list checked-variable)
      )
      (not (null? new-domain))
    )
  )
)

;;;;;;;;;;;;;;;
;; add to the todo list all constraints that need to be rechecked because of checked-variable's domain changing
;;
;; @param constraint-network = a constraint network made with (new-constraint-network)
;; @param todo-list = a list of todo-items, describing the work that still needs to be done.
;; @param checked-variable = the variable whose domain changed
;; @returns Undefined
;; @sideeffect modifies todo-list (adds new work)
(define (propagate-constraints::cascade! constraint-network todo-list checked-variable)
  (let ((to-append 
            (append
              ;; we need to backward-recheck all constraint-instances with checked-variable at their head
              (map-and-filter-false
                (lambda (constraint-instance)
                  (if (equal? (constraint-instance::head constraint-instance) checked-variable)
                    (let ((new-item (propagate-constraints::new-todo-item constraint-instance 'backward))) 
                      (if (contains? todo-list new-item)
                        #f
                        new-item
                        )
                      )               
                    #f
                  )
                )
                (skip-list-label (constraint-network::constraint-instances constraint-network))
              )
    
              ;; we need to forward-recheck all constraint-instances with checked-variable at their tail
              (map-and-filter-false
                (lambda (constraint-instance)
                  (if (equal? (constraint-instance::tail constraint-instance) checked-variable)
                    (let ((new-item (propagate-constraints::new-todo-item constraint-instance 'forward))) 
                      (if (contains? todo-list new-item)
                        #f
                        new-item
                        )
                      )
                   #f
                   )
                )
                (skip-list-label (constraint-network::constraint-instances constraint-network))
              )
            )))
  (set-cdr! todo-list (append to-append (skip-list-label todo-list)))
  )
)

          
    


