;;;; -*- mode:Scheme -*- ;;;;

; =========================================================
; ========        SCHEME UTILITY FUNCTIONS      ===========
; =========================================================

;===========================================================
;                       DEBUGGING
;===========================================================
;;; Most of these exist in MIT Scheme but are here for portability...
(define *t:silent* #f)
(define (display* . l)
  ;; Print the list of arguments
  (cond (*t:silent* #f)
	(else
	 (for-each display l)
	 (newline))))


;===========================================================
;                       LIST ACCESS
;===========================================================
;;; Most of these exist in MIT Scheme but are here for portability...

(define first car)
(define second cadr)
(define third caddr)
(define fourth cadddr)
(define (fifth x) (cadddr (cdr x)))
(define (sixth x) (cadddr (cddr x)))
(define (seventh x) (cadddr (cdddr x)))
(define (eighth x) (cadddr (cddddr x))) 
(define rest cdr)
(define skip-list-label cdr)
(define (random-list-entry l)
  (if (null? l)
      (error 'random-index "called with empty list")
      (list-ref l (random (length l)))))

;===========================================================
;                      LIST MAPPING
;===========================================================
(define (append-map fn . x)
  (if (any? x null?)
      '()
      (let ((val (apply fn (map car x))))
	(if (list? val)
	    (append val (apply append-map (cons fn (map cdr x))))
	    (error "append-map expects list values, it got this:" val)))))

;===========================================================
;                      LIST FILTERING
;===========================================================
;; (delete-matching-items list pred) returns a newly allocated copy of 
;; list containing only the elements for which predicate is false. 
;; Predicate must be a procedure of one argument.
(define delete-matching-items list-transform-negative)

(define keep-matching-items list-transform-positive)

(define (map-and-filter-false fn list)
  (if (null? list)
      '()
      (let ((answer (fn (first list))))
	(if answer
	    (cons answer (map-and-filter-false fn (rest list)))
	    (map-and-filter-false fn (rest list))))))

(define (remove-if predicate elements)
  "
  Purpose:	Remove all list elements that satisfy predicate
  "
  (cond ((null? elements) '())
	((predicate (first elements))
	 (remove-if predicate (rest elements)))
	(else
	 (cons (first elements)
	       (remove-if predicate (rest elements))))))

(define (remove-if-not predicate elements)
  "
  Purpose:	Remove all list elements that DO NOT satisfy predicate
  "
  (remove-if (lambda (x) (not (predicate x))) elements))



;===========================================================
;                      LIST SEARCH
;===========================================================

;; returns #t iff, for all items in the list, (fn item) is true
(define (for-all? x fn)
  (if (null? x) #t
      (if (fn (car x)) 
	  (for-all? (cdr x) fn)
	  #f)))

;; returns #t iff, for some item in the list, (fn item) is true
(define (any? x fn)
  (if (null? x) #f
    (if (fn (car x)) 
       #t
	    (any? (cdr x) fn))))

;; predicate to search of an element in a list
;; @param list : a scheme list in which to be searched
;; @param element : the object to search for.
;; @returns #t if any x in the list satisfies (equal? x element), else #f
(define (contains? list element) 
  (any? list (lambda (x) (equal? x element)))
)

;; returns some item in the list for which (fn item) is true, or #f if no such
;; item exists
(define (get-any x fn)
  (if (null? x) #f
      (if (fn (car x))
	      (car x)
	      (there-exists? (cdr x) fn))))

;;; Finds the position (integer from 0) of item in the list.  The test
;;; is eq?.  If a third arg is provided then that function is applied
;;; to the list element before testing for equality.
;;; (position 'b '(a b c)) => 1
;;; (position 'b '((a 1) (b 2) (2 3)) first) => 1
(define (position item list . args)
  (let ((found #f)
	(key-fn
	 (if (null? args)
	     #f
	     (first args))))
    (do ((x list (rest x))
	 (i 0 (+ i 1)))
	((or (null? x) found))
      (if (equal? (if key-fn (key-fn (first x)) (first x)) item)
	  (set! found i)))
    found))


;===========================================================
;                      LIST COPY
;===========================================================

;; recursively clones a scheme data structure
(define (clone x) 
  (if (list? x) (map clone x) x)
)


;===========================================================
;                      SET OPERATIONS
;===========================================================

;; computes a - b, where a and b are sets
;; @param a : a scheme list, with each element appearing at most once
;; @param b : a scheme list, with each element appearing at most once
;; @returns : a scheme list, including all the elements in a which are not also in b
(define (set-subtract a b)
  (delete-matching-items a (lambda (x) (contains? b x)))
)

;; computes a UNION b, where a and b are sets
;; @param a : a scheme list, with each element appearing at most once
;; @param b : a scheme list, with each element appearing at most once
;; @returns : a scheme list, including all the elements in a or in b.  The ordering
;;            of these elements is left unspecified.
(define (set-union a b) 
  (append (set-subtract a b) b)
)

;; computes the union of all the elements in the-list
;; @param the-list : a scheme list, 
;;    whose elements are each : sets, that is, scheme lists with each element appearing 
;;                              at most once
;; @returns : a scheme list, with each element appearing at most once, and including every
;;            element that was included in any of the-lists' element lists.  Thus, 
;;              (condense-list '((a b) (c) (b d))) 
;;            evaluates to some unspecified permutation of
;;              '(a b c d)
(define (condense-list the-list) (reduce set-union '() the-list))



;============================================================
;                        INDEX LISTS
;============================================================

;;; Constructs a list of indices starting at 0 that goes up to n-1
;;; where n is the length of the list l.
;;; (index-list '(a b c)) => (0 1 2)
(define (index-list l)
  (define (index-list-aux l index)
    (if (null? l)
	'()
	(cons index (index-list-aux (rest l) (+ index 1)))))
  (index-list-aux l 0))

(define (make-index-list n)
  (define (make-index-list-aux i)
    (if (< i n) 
	(cons i (make-index-list-aux (+ 1 i)))
	'()))
  (make-index-list-aux 0))




